<?php
/**
 * Auto Project Cleanup & Centralization Tool
 * Author: Avinash Babu Custom Version
 */

$baseDir = __DIR__;
$trashDir = $baseDir . '/trash';
$assetsDir = $baseDir . '/assets';
$allowedRootFiles = ['index.php', 'index.html'];

// Create trash & assets folders
if (!is_dir($trashDir)) mkdir($trashDir, 0755, true);
if (!is_dir($assetsDir)) mkdir($assetsDir, 0755, true);

// Extensions
$fileExts = ['php', 'html', 'css', 'js'];
$imageExts = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'webp'];

// Collect all project file content
$allContent = '';
$rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($baseDir, FilesystemIterator::SKIP_DOTS));
foreach ($rii as $file) {
    if ($file->isFile() && in_array(strtolower(pathinfo($file, PATHINFO_EXTENSION)), $fileExts)) {
        $allContent .= file_get_contents($file->getPathname()) . "\n";
    }
}

/**
 * Move unused files/folders to trash
 */
function move_unused_to_trash($baseDir, $trashDir, $allowedRootFiles, $fileExts, $allContent) {
    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($baseDir, FilesystemIterator::SKIP_DOTS), RecursiveIteratorIterator::SELF_FIRST);
    foreach ($rii as $file) {
        $path = $file->getPathname();
        if (strpos($path, $trashDir) !== false) continue;
        if ($file->isDir()) {
            // Skip root folder itself
            if ($path === $baseDir) continue;
            // Skip if contains used files
            $used = false;
            foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS)) as $innerFile) {
                if ($innerFile->isFile()) {
                    $ext = strtolower(pathinfo($innerFile->getFilename(), PATHINFO_EXTENSION));
                    if (in_array($ext, $fileExts) && stripos($allContent, $innerFile->getFilename()) !== false) {
                        $used = true; break;
                    }
                }
            }
            if (!$used) {
                rename($path, $trashDir . '/' . basename($path) . '_' . time());
            }
        } else {
            // Root file check
            if (dirname($path) === $baseDir && !in_array(basename($path), $allowedRootFiles)) {
                rename($path, $trashDir . '/' . basename($path));
            }
        }
    }
}

/**
 * Centralize used assets into /assets/
 */
function centralize_assets($baseDir, $assetsDir, $allContent, $imageExts) {
    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($baseDir, FilesystemIterator::SKIP_DOTS));
    foreach ($rii as $file) {
        if ($file->isFile()) {
            $ext = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));
            if (in_array($ext, $imageExts) || in_array($ext, ['css','js'])) {
                if (stripos($allContent, $file->getFilename()) !== false) {
                    $subFolder = 'others';
                    if (in_array($ext, ['css'])) $subFolder = 'css';
                    elseif (in_array($ext, ['js'])) $subFolder = 'js';
                    elseif (strpos(strtolower($file->getFilename()), 'logo') !== false) $subFolder = 'images/logo';
                    elseif (strpos(strtolower($file->getFilename()), 'icon') !== false) $subFolder = 'images/icons';
                    elseif (strpos(strtolower($file->getFilename()), 'banner') !== false) $subFolder = 'images/banner';
                    else $subFolder = 'images/others';

                    $destFolder = $assetsDir . '/' . $subFolder;
                    if (!is_dir($destFolder)) mkdir($destFolder, 0755, true);

                    $destPath = $destFolder . '/' . $file->getFilename();
                    @rename($file->getPathname(), $destPath);

                    update_paths_in_files($baseDir, $file->getFilename(), 'assets/' . $subFolder . '/' . $file->getFilename());
                }
            }
        }
    }
}

/**
 * Update all file paths in project
 */
function update_paths_in_files($baseDir, $oldName, $newPath) {
    $exts = ['php','html','css','js'];
    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($baseDir, FilesystemIterator::SKIP_DOTS));
    foreach ($rii as $file) {
        if ($file->isFile()) {
            $ext = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));
            if (in_array($ext, $exts)) {
                $content = file_get_contents($file->getPathname());
                if (strpos($content, $oldName) !== false) {
                    $content = preg_replace(
                        '#(["\'])[^"\']*' . preg_quote($oldName, '#') . '(["\'])#',
                        '$1' . $newPath . '$2',
                        $content
                    );
                    file_put_contents($file->getPathname(), $content);
                }
            }
        }
    }
}

// Run cleanup
move_unused_to_trash($baseDir, $trashDir, $allowedRootFiles, $fileExts, $allContent);
centralize_assets($baseDir, $assetsDir, $allContent, $imageExts);

echo "✅ Cleanup complete! Unused files moved to 'trash', assets centralized.\n";
