<?php
// editprofile.php
session_start();
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: login");
    exit;
}
require_once "../config.php";

$success_message = $error_message = "";

/* ==============================================================
   1) PROFILE AVATAR UPDATE (existing)
============================================================== */
if ($_POST && isset($_POST['profile_image'])) {
    $profileImage = intval($_POST['profile_image']);
    if ($profileImage >= 1 && $profileImage <= 19) {
        $stmt = $conn->prepare("UPDATE users SET profile_image = ? WHERE username = ?");
        $stmt->bind_param("is", $profileImage, $_SESSION['username']);
        if ($stmt->execute()) {
            $success_message = "Profile image updated successfully!";
        } else {
            $error_message = "Failed to update profile image.";
        }
        $stmt->close();
    } else {
        $error_message = "Invalid profile image selected.";
    }
}

/* ==============================================================
   2) USERNAME CHANGE
============================================================== */
if ($_POST && !empty($_POST['new_username'])) {
    $newUsername = trim($_POST['new_username']);
    // quick validations
    if (strlen($newUsername) < 3 || strlen($newUsername) > 20) {
        $error_message = "Username must be 3–20 characters.";
    } elseif (!preg_match('/^[A-Za-z0-9_]+$/', $newUsername)) {
        $error_message = "Only letters, numbers and underscore allowed.";
    } else {
        // check blacklist
        $stmt = $conn->prepare("SELECT 1 FROM username_blacklist WHERE LOWER(word)=LOWER(?) LIMIT 1");
        $stmt->bind_param("s", $newUsername);
        $stmt->execute();
        $stmt->store_result();
        if ($stmt->num_rows) {
            $error_message = "This username is not allowed.";
        } else {
            $stmt->close();
            // check uniqueness
            $stmt = $conn->prepare("SELECT 1 FROM users WHERE LOWER(username)=LOWER(?) AND username<>? LIMIT 1");
            $stmt->bind_param("ss", $newUsername, $_SESSION['username']);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows) {
                $error_message = "Username already taken.";
            } else {
                $stmt->close();
                // update
                $stmt = $conn->prepare("UPDATE users SET username = ? WHERE username = ?");
                $stmt->bind_param("ss", $newUsername, $_SESSION['username']);
                if ($stmt->execute()) {
                    $_SESSION['username'] = $newUsername; // keep session in sync
                    $success_message = "Username updated successfully!";
                } else {
                    $error_message = "Could not update username.";
                }
                $stmt->close();
            }
        }
    }
}

/* ==============================================================
   3) PASSWORD CHANGE
============================================================== */
if ($_POST && !empty($_POST['current_password'])) {
    $current = $_POST['current_password'];
    $new     = $_POST['new_password'];
    $confirm = $_POST['confirm_password'];

    if ($new !== $confirm) {
        $error_message = "New passwords do not match.";
    } elseif (strlen($new) < 6) {
        $error_message = "Password must be at least 6 characters.";
    } else {
        // fetch current hash
        $stmt = $conn->prepare("SELECT password FROM users WHERE username = ?");
        $stmt->bind_param("s", $_SESSION['username']);
        $stmt->execute();
        $stmt->bind_result($hash);
        $stmt->fetch();
        if (!password_verify($current, $hash)) {
            $error_message = "Current password is incorrect.";
        } else {
            $stmt->close();
            $newHash = password_hash($new, PASSWORD_DEFAULT);
            $stmt = $conn->prepare("UPDATE users SET password = ? WHERE username = ?");
            $stmt->bind_param("ss", $newHash, $_SESSION['username']);
            if ($stmt->execute()) {
                $success_message = "Password changed successfully!";
            } else {
                $error_message = "Could not change password.";
            }
            $stmt->close();
        }
    }
}

/* Fetch current user data */
$userQuery = "SELECT profile_image FROM users WHERE username = ? LIMIT 1";
$stmt = $conn->prepare($userQuery);
$stmt->bind_param("s", $_SESSION['username']);
$stmt->execute();
$userRow = $stmt->get_result()->fetch_assoc();
$currentProfileImage = $userRow['profile_image'] ?: 1;
$stmt->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Profile</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        /* === existing styles === */
        :root {
            --primary: #fbb034;
            --secondary: #fbb034;
            --dark-bg: #0c0c0f;
            --card-bg: #131317;
            --text-light: #e0e0e0;
            --text-muted: #aaa;
        }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
            font-family:'Montserrat',sans-serif;
            background:var(--dark-bg);
            color:var(--text-light);
            margin:0;
            padding:80px 20px 80px;
            min-height:100vh;
        }
        header {
            display:flex;justify-content:space-between;align-items:center;
            padding:15px 20px;
            background:linear-gradient(135deg,rgba(30,30,35,.95) 0%,rgba(20,20,25,.95) 100%);
            backdrop-filter:blur(10px);
            position:fixed;top:0;left:0;right:0;z-index:1000;
            border-bottom:1px solid rgba(255,255,255,.1);
            height:60px;box-shadow:0 5px 20px rgba(0,0,0,.3);
        }
        .back-btn {
            color:var(--primary);font-size:1.2rem;cursor:pointer;
            transition:transform .3s ease;
        }
        .back-btn:hover { transform:translateX(-5px); }
        .page-title { font-size:1.2rem;font-weight:600;color:var(--text-light); }
        .container { max-width:600px;margin:0 auto; }
        .profile-section {
            background:var(--card-bg);border-radius:16px;padding:30px;margin-bottom:20px;
            border:1px solid rgba(255,255,255,.1);
        }
        .current-profile { text-align:center;margin-bottom:30px; }
        .current-avatar {
            width:80px;height:80px;border-radius:50%;border:3px solid var(--primary);
            margin-bottom:15px;
        }
        .profile-grid {
            display:grid;grid-template-columns:repeat(auto-fill,minmax(80px,1fr));
            gap:15px;margin-top:20px;
        }
        .profile-option {
            text-align:center;cursor:pointer;transition:all .3s ease;
            padding:10px;border-radius:12px;border:2px solid transparent;
        }
        .profile-option:hover {
            background:rgba(255,255,255,.05);transform:translateY(-3px);
        }
        .profile-option.selected {
            border-color:var(--primary);background:rgba(251,176,52,.1);
        }
        .profile-option img {
            width:60px;height:60px;border-radius:50%;border:2px solid rgba(255,255,255,.1);
            transition:all .3s ease;
        }
        .profile-option.selected img { border-color:var(--primary); }
        .profile-number { font-size:.75rem;color:var(--text-muted);margin-top:5px; }
        .save-btn {
            width:100%;padding:15px;
            background:linear-gradient(135deg,var(--primary) 0%,#f7971e 100%);
            color:#000;border:none;border-radius:12px;font-size:1rem;font-weight:600;
            cursor:pointer;transition:all .3s ease;margin-top:30px;
        }
        .save-btn:hover {
            transform:translateY(-2px);box-shadow:0 10px 20px rgba(251,176,52,.3);
        }
        .save-btn:disabled {
            opacity:.6;cursor:not-allowed;transform:none;box-shadow:none;
        }
        .message {
            padding:15px;border-radius:8px;margin-bottom:20px;text-align:center;
        }
        .success { background:rgba(34,197,94,.1);border:1px solid rgba(34,197,94,.3);color:#22c55e; }
        .error   { background:rgba(239,68,68,.1);border:1px solid rgba(239,68,68,.3);color:#ef4444; }

        /* form fields */
        .form-group { margin-bottom:18px; }
        label { display:block;margin-bottom:6px;font-size:.9rem;color:var(--text-muted); }
        input[type=text], input[type=password] {
            width:100%;padding:12px;border-radius:8px;border:1px solid rgba(255,255,255,.1);
            background:rgba(255,255,255,.05);color:var(--text-light);font-size:1rem;
        }
        input::placeholder { color:var(--text-muted); }

        @media(max-width:768px){
            .profile-grid { grid-template-columns:repeat(4,1fr);gap:10px; }
            .profile-option img { width:50px;height:50px; }
            .current-avatar { width:70px;height:70px; }
        }
    </style>
</head>
<body>

<header>
    <div class="back-btn" onclick="window.location.href='/game/profile'">
        <i class="fas fa-arrow-left"></i>
    </div>
    <div class="page-title">Edit Profile</div>
    <div></div>
</header>

<div class="container">
    <?php if (isset($success_message)): ?>
        <div class="message success"><i class="fas fa-check-circle"></i> <?= $success_message ?></div>
    <?php endif; ?>
    <?php if (isset($error_message)): ?>
        <div class="message error"><i class="fas fa-exclamation-circle"></i> <?= $error_message ?></div>
    <?php endif; ?>

    <!-- ===================== AVATAR SECTION ===================== -->
    <div class="profile-section">
        <h3 style="margin-bottom:20px;text-align:center;">Choose Profile Avatar</h3>
        <div class="current-profile">
            <img id="selectedAvatar" src="/images/profile/<?= $currentProfileImage ?>.png" class="current-avatar">
            <div style="font-size:.9rem;color:var(--text-muted);">Current Profile (#<?= $currentProfileImage ?>)</div>
        </div>

        <form method="POST" id="profileForm">
            <input type="hidden" name="profile_image" id="selectedProfileInput" value="<?= $currentProfileImage ?>">
            <div class="profile-grid">
                <?php for ($i = 1; $i <= 19; $i++): ?>
                    <div class="profile-option <?= ($i == $currentProfileImage) ? 'selected' : '' ?>" onclick="selectProfile(<?= $i ?>)">
                        <img src="/images/profile/<?= $i ?>.png" alt="Profile <?= $i ?>">
                        <div class="profile-number">#<?= $i ?></div>
                    </div>
                <?php endfor; ?>
            </div>
            <button type="submit" class="save-btn" id="saveBtn">
                <i class="fas fa-save"></i> Save Profile
            </button>
        </form>
    </div>

    <!-- ===================== USERNAME SECTION ===================== -->
    <div class="profile-section">
        <h3 style="margin-bottom:20px;text-align:center;">Change Username</h3>
        <form method="POST">
            <div class="form-group">
                <label>Current Username</label>
                <input type="text" value="<?= htmlspecialchars($_SESSION['username']) ?>" readonly>
            </div>
            <div class="form-group">
                <label>New Username</label>
                <input type="text" name="new_username" placeholder="3-20 characters" required>
            </div>
            <button type="submit" class="save-btn"><i class="fas fa-user-edit"></i> Update Username</button>
        </form>
    </div>

    <!-- ===================== PASSWORD SECTION ===================== -->
    <div class="profile-section">
        <h3 style="margin-bottom:20px;text-align:center;">Change Password</h3>
        <form method="POST">
            <div class="form-group">
                <label>Current Password</label>
                <input type="password" name="current_password" placeholder="••••••••" required>
            </div>
            <div class="form-group">
                <label>New Password</label>
                <input type="password" name="new_password" placeholder="Min 6 characters" required>
            </div>
            <div class="form-group">
                <label>Confirm New Password</label>
                <input type="password" name="confirm_password" placeholder="Repeat new password" required>
            </div>
            <button type="submit" class="save-btn"><i class="fas fa-key"></i> Change Password</button>
        </form>
    </div>
</div>

<script>
function selectProfile(profileNumber){
    document.querySelectorAll('.profile-option').forEach(el=>el.classList.remove('selected'));
    event.currentTarget.classList.add('selected');
    document.getElementById('selectedAvatar').src = `/images/profile/${profileNumber}.png`;
    document.getElementById('selectedProfileInput').value = profileNumber;
    const saveBtn = document.getElementById('saveBtn');
    const current = <?= $currentProfileImage ?>;
    if(profileNumber !== current){
        saveBtn.disabled=false;
        saveBtn.innerHTML='<i class="fas fa-save"></i> Save Changes';
    }else{
        saveBtn.disabled=true;
        saveBtn.innerHTML='<i class="fas fa-check"></i> Current Profile';
    }
}
document.addEventListener('DOMContentLoaded',()=>{
    document.getElementById('saveBtn').disabled=true;
    document.getElementById('saveBtn').innerHTML='<i class="fas fa-check"></i> Current Profile';
});
</script>
</body>
</html>