<?php
// API-based Game History Page
// This page fetches game history from the API and displays it

// Initialize session
session_start();

// Check if user is logged in
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: login");
    exit;
}

require_once "../config.php";

// User ID from session
$userId = $_SESSION['id'];
$username = $_SESSION['username'];

// API Configuration
$API_BASE_URL = "https://api.inout.games/api";
$OPERATOR_ID = "ee2013ed-e1f0-4d6e-97d2-f36619e2eb52";

// Function to fetch game history from API
function fetchGameHistoryFromAPI($userId, $limit = 50) {
    global $API_BASE_URL, $OPERATOR_ID;
    
    // In a real implementation, this would make API calls
    // For now, we'll simulate with sample data structure
    
    $sampleData = [
        [
            'id' => 1,
            'game_name' => 'Chicken Road',
            'bet_amount' => 100.00,
            'win_amount' => 250.00,
            'result' => 'win',
            'multiplier' => 2.5,
            'played_at' => '2024-01-15 10:30:00',
            'game_details' => 'Crossed 5 roads successfully',
            'status' => 'completed'
        ],
        [
            'id' => 2,
            'game_name' => 'Crash Game',
            'bet_amount' => 50.00,
            'win_amount' => 0.00,
            'result' => 'loss',
            'multiplier' => 0.0,
            'played_at' => '2024-01-15 11:45:00',
            'game_details' => 'Crashed at 1.5x',
            'status' => 'completed'
        ],
        [
            'id' => 3,
            'game_name' => 'Dice Roll',
            'bet_amount' => 25.00,
            'win_amount' => 75.00,
            'result' => 'win',
            'multiplier' => 3.0,
            'played_at' => '2024-01-15 14:20:00',
            'game_details' => 'Rolled 6',
            'status' => 'completed'
        ]
    ];
    
    return $sampleData;
}

// Fetch game history
$gameHistory = fetchGameHistoryFromAPI($userId);

// Calculate statistics
$totalGames = count($gameHistory);
$totalWins = count(array_filter($gameHistory, fn($g) => $g['result'] === 'win'));
$totalLosses = count(array_filter($gameHistory, fn($g) => $g['result'] === 'loss'));
$totalBet = array_sum(array_column($gameHistory, 'bet_amount'));
$totalWon = array_sum(array_column($gameHistory, 'win_amount'));
$winRate = $totalGames > 0 ? round(($totalWins / $totalGames) * 100, 2) : 0;
$profit = $totalWon - $totalBet;
$avgMultiplier = $totalGames > 0 ? round(array_sum(array_column($gameHistory, 'multiplier')) / $totalGames, 2) : 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Game History - Chicken Road</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        :root {
            --primary: #fbb034;
            --secondary: #00ffcc;
            --dark-bg: #0c0c0f;
            --card-bg: #131317;
            --text-light: #e0e0e0;
            --text-muted: #aaa;
            --success: #00ff88;
            --danger: #ff4757;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Montserrat', sans-serif;
            background: var(--dark-bg);
            color: var(--text-light);
            margin: 0;
            padding: 80px 0 80px;
            min-height: 100vh;
        }

        header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: linear-gradient(135deg, rgba(30, 30, 35, 0.95) 0%, rgba(20, 20, 25, 0.95) 100%);
            backdrop-filter: blur(10px);
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            height: 60px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.3);
        }

        .logo {
            font-size: 1.4rem;
            font-weight: 700;
            color: var(--primary);
            text-shadow: 0 0 10px rgba(251, 176, 52, 0.3);
        }

        .back-btn {
            color: var(--text-light);
            font-size: 1.2rem;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .back-btn:hover {
            color: var(--primary);
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .stats-section {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: var(--card-bg);
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            border: 1px solid rgba(255, 255, 255, 0.1);
            transition: transform 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-value {
            font-size: 2rem;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .stat-label {
            color: var(--text-muted);
            font-size: 0.9rem;
        }

        .win-rate { color: var(--success); }
        .profit { color: var(--primary); }
        .loss { color: var(--danger); }

        .game-list {
            background: var(--card-bg);
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
        }

        .game-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            transition: background 0.3s ease;
        }

        .game-item:hover {
            background: rgba(255, 255, 255, 0.05);
        }

        .game-info {
            flex: 1;
        }

        .game-name {
            font-weight: bold;
            color: var(--primary);
            margin-bottom: 5px;
        }

        .game-details {
            font-size: 0.9rem;
            color: var(--text-muted);
        }

        .game-stats {
            text-align: right;
        }

        .bet-amount {
            color: var(--danger);
            font-weight: bold;
        }

        .win-amount {
            color: var(--success);
            font-weight: bold;
        }

        .multiplier {
            color: var(--secondary);
            font-weight: bold;
        }

        .result-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: bold;
        }

        .result-win {
            background: var(--success);
            color: var(--dark-bg);
        }

        .result-loss {
            background: var(--danger);
            color: white;
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--text-muted);
        }

        .empty-state i {
            font-size: 4rem;
            margin-bottom: 20px;
            opacity: 0.5;
        }

        @media screen and (max-width: 768px) {
            .stats-section {
                grid-template-columns: 1fr;
            }
            
            .game-item {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .game-stats {
                text-align: left;
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>

<header>
    <div class="back-btn" onclick="window.location.href='me.html'">
        <i class="fas fa-arrow-left"></i>
    </div>
    <div class="logo">API Game History</div>
    <div></div>
</header>

<div class="container">
    <!-- Statistics Section -->
    <div class="stats-section">
        <div class="stat-card">
            <div class="stat-value win-rate"><?php echo $totalGames; ?></div>
            <div class="stat-label">Total Games</div>
        </div>
        <div class="stat-card">
            <div class="stat-value win-rate"><?php echo $winRate; ?>%</div>
            <div class="stat-label">Win Rate</div>
        </div>
        <div class="stat-card">
            <div class="stat-value <?php echo $profit >= 0 ? 'profit' : 'loss'; ?>">₹<?php echo number_format(abs($profit), 2); ?></div>
            <div class="stat-label"><?php echo $profit >= 0 ? 'Profit' : 'Loss'; ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-value win-rate"><?php echo $avgMultiplier; ?>x</div>
            <div class="stat-label">Avg Multiplier</div>
        </div>
    </div>

    <!-- Game History List -->
    <div class="game-list">
        <?php if (!empty($gameHistory)): ?>
            <?php foreach ($gameHistory as $game): ?>
                <div class="game-item">
                    <div class="game-info">
                        <div class="game-name"><?php echo htmlspecialchars($game['game_name']); ?></div>
                        <div class="game-details">
                            <?php echo date('M d, Y H:i', strtotime($game['played_at'])); ?>
                        </div>
                    </div>
                    <div class="game-stats">
                        <div class="bet-amount">₹<?php echo number_format($game['bet_amount'], 2); ?></div>
                        <div class="win-amount">₹<?php echo number_format($game['win_amount'], 2); ?></div>
                        <div class="multiplier"><?php echo $game['multiplier']; ?>x</div>
                        <span class="result-badge result-<?php echo $game['result']; ?>">
                            <?php echo ucfirst($game['result']); ?>
                        </span>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <div class="empty-state">
                <i class="fas fa-gamepad"></i>
                <h3>No Game History Found</h3>
                <p>Start playing games to see your history here!</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// API integration functions
function fetchGameHistoryFromAPI() {
    // In a real implementation, this would make API calls
    console.log('Fetching game history from API...');
}

// Auto refresh every 30 seconds
setInterval(() => {
    fetchGameHistoryFromAPI();
}, 30000);
</script>

</body>
</html>
<?php
// Close database connection if needed
// $conn->close();
?>
