<?php
declare(strict_types=1);

// Enable error reporting (disable in production)
error_reporting(E_ALL);
ini_set('display_errors', '1');

session_start();
if (empty($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("Location: login.php");
    exit;
}

require_once "../config.php";

// Database connection check
if ($conn->connect_error) {
    die("Database connection failed: " . htmlspecialchars($conn->connect_error));
}

$userId = (int)$_SESSION['id'];
$message = "";
$claimedAmount = 0.0;
$claimedCode = '';
$messageType = ''; // 'success' or 'error'

// Handle Gift Claim
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['gift_code'])) {
    $code = trim($_POST['gift_code']);

    // Fetch gift info
    $stmt = $conn->prepare("
        SELECT code, amount, is_active 
        FROM gift 
        WHERE code = ? AND is_active = 1 
        LIMIT 1
    ");
    $stmt->bind_param("s", $code);
    $stmt->execute();
    $gift = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$gift) {
        $message = "Gift code is invalid or expired!";
        $messageType = 'error';
    } else {
        // Check if already claimed
        $stmt = $conn->prepare("
            SELECT id 
            FROM claimed_gifts 
            WHERE user_id = ? AND code = ? 
            LIMIT 1
        ");
        $stmt->bind_param("is", $userId, $code);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows > 0) {
            $message = "You have already claimed this gift code!";
            $messageType = 'error';
        } else {
            $amount = (float)$gift['amount'];

            // Update balance
            $stmtUp = $conn->prepare("
                UPDATE users 
                SET balance = balance + ? 
                WHERE id = ?
            ");
            $stmtUp->bind_param("di", $amount, $userId);
            $stmtUp->execute();
            $stmtUp->close();

            // Save claim record
            $stmtIns = $conn->prepare("
                INSERT INTO claimed_gifts (user_id, code) 
                VALUES (?, ?)
            ");
            $stmtIns->bind_param("is", $userId, $code);
            $stmtIns->execute();
            $stmtIns->close();

            // Add transaction record
            $desc = "Gift Claimed: " . $code;
            $stmtTxn = $conn->prepare("
                INSERT INTO transactions (user_id, type, amount, description) 
                VALUES (?, 'gift', ?, ?)
            ");
            $stmtTxn->bind_param("ids", $userId, $amount, $desc);
            $stmtTxn->execute();
            $stmtTxn->close();

            $claimedAmount = $amount;
            $claimedCode = $code;
            $message = "Gift code {$code} claimed! ₹" . number_format($amount, 2) . " added to your wallet.";
            $messageType = 'success';
        }
        $stmt->close();
    }
}

// Fetch claim history
$stmtHist = $conn->prepare("
    SELECT cg.code, g.amount, cg.claimed_at
    FROM claimed_gifts cg
    LEFT JOIN gift g 
      ON cg.code COLLATE utf8mb4_unicode_ci = g.code COLLATE utf8mb4_unicode_ci
    WHERE cg.user_id = ?
    ORDER BY cg.claimed_at DESC
");
$stmtHist->bind_param("i", $userId);
$stmtHist->execute();
$historyResult = $stmtHist->get_result();
$stmtHist->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Gift Center</title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<style>
:root {
    --primary: #fbb034;
    --dark-bg: #0c0c0f;
    --card-bg: #131317;
    --gold-gradient: linear-gradient(90deg,#fbb034,#ffd700,#f7971e);
    --text-light: #e0e0e0;
    --text-muted: #aaa;
}
body {
    background: var(--dark-bg);
    font-family: 'Montserrat', sans-serif;
    margin:0;
    padding:72px 0 40px;
    color: var(--text-light);
}
header {
    position: fixed; top:0; left:0; right:0; height: 60px;
    background: rgba(19,19,23,0.95);
    border-bottom: 1px solid rgba(255,255,255,0.1);
    display:flex; align-items:center;
    padding: 0 20px;
    z-index: 1000;
}
header a { color: var(--primary); text-decoration:none; font-weight:bold; display:flex; align-items:center; }
header a i { margin-right:6px; }
.gift-container, .history-section {
    max-width: 400px;
    margin:auto;
    border-radius:14px;
    background: var(--card-bg);
    box-shadow: 0 6px 20px rgba(0,0,0,0.17);
}
.gift-container { padding-bottom:20px; }
.gift-banner { text-align:center; padding:18px 0; }
.gift-banner img { width:120px; margin-bottom: 8px; }
.gift-banner h2 { color:var(--primary); font-size:1.3rem; font-weight:700; }
.gift-card { padding:0 18px; }
.form-label { text-align:left; color:var(--primary); font-size:0.96rem; font-weight:600; }
.gift-form input {
    width:100%; padding:13px; border-radius:10px; border:none;
    background:rgba(255,255,255,0.07); color: var(--text-light);
    margin:7px 0 12px; font-size:1rem;
}
.gift-form button {
    background: var(--gold-gradient); border:none; border-radius:20px; color:#232323;
    font-weight:600; font-size:1.08rem; padding:12px 0;
    width:100%; cursor:pointer; transition: background 0.2s;
}
.gift-form button:hover { background:linear-gradient(90deg,#ffd700,#fbb034,#f7971e); }
.history-section { margin-top:20px; padding:11px 18px 20px; }
.history-title { color:var(--primary); font-weight:600; margin-bottom:11px; display:flex; align-items:center; gap:8px; }
.history-row { display:flex; align-items:center; gap:10px; border-bottom:1px solid rgba(255,255,255,0.04); padding:10px 0; }
.history-code { background: var(--gold-gradient); color:#232323; font-weight:700; padding:2px 12px; border-radius:100px; font-size:0.94rem; }
.history-amt { color:var(--primary); font-weight:500; }
.history-date { color:var(--text-muted); font-size:0.82rem; margin-left:auto; }
.history-empty { text-align:center; color:var(--text-muted); padding:20px 0; }
.modal { display:none; position:fixed; z-index:2000; left:0; top:0; width:100%; height:100%; background:rgba(14,14,18,0.85); }
.modal-content {
    background:var(--card-bg); border:2px solid var(--primary); border-radius:14px; max-width:340px;
    margin:15% auto; padding:24px 20px; text-align:center; position:relative; animation: fadeIn 0.3s ease-out;
}
.modal-content i { font-size:2.4rem; margin-bottom:8px; }
.modal-content.success i { color:limegreen; }
.modal-content.error i { color:red; }
.close { position:absolute; right:16px; top:13px; font-size:22px; cursor:pointer; color:var(--text-muted); }
.close:hover { color:var(--primary); }
@keyframes fadeIn { from { opacity:0; transform:translateY(-20px);} to{opacity:1; transform:translateY(0);} }
@media only screen and (max-width: 480px) {
    body { padding: 65px 0 30px; }
    .gift-container, .history-section { max-width: 92vw; margin: 10px auto; border-radius: 10px; box-shadow: none; }
    .gift-banner img { width: 90px; }
    .gift-banner h2 { font-size: 1.15rem; }
    .form-label { font-size: 0.9rem; }
    .gift-form input { font-size: 0.9rem; padding: 11px; }
    .gift-form button { font-size: 1rem; padding: 11px 0; border-radius: 16px; }
    .history-title { font-size: 0.95rem; }
    .history-code { font-size: 0.85rem; padding: 2px 8px; }
    .history-amt { font-size: 0.85rem; }
    .history-date { font-size: 0.78rem; }
    .modal-content { width: 90%; margin: 40% auto; font-size: 0.9rem; padding: 18px 15px; }
    .modal-content i { font-size: 2rem; }
    .close { top: 8px; right: 10px; font-size: 20px; }
}
</style>
</head>
<body>

<header>
    <a href="profile.php"><i class="fas fa-arrow-left"></i>Back</a>
</header>

<div class="gift-container">
    <div class="gift-banner">
        <img src="https://bdg55.com/assets/png/gift-a8f321e1.webp" alt="Gift" />
        <h2>We have a gift for you</h2>
    </div>
    <div class="gift-card">
        <form class="gift-form" method="post">
            <div class="form-label">Please enter the gift code below</div>
            <input type="text" name="gift_code" placeholder="Enter gift code" required>
            <button type="submit">Receive</button>
        </form>
    </div>
</div>

<div class="history-section">
    <div class="history-title"><i class="fas fa-history"></i> History</div>
    <?php if($historyResult && $historyResult->num_rows > 0): ?>
        <?php while($row = $historyResult->fetch_assoc()): ?>
            <div class="history-row">
                <div class="history-code"><?= htmlspecialchars($row['code']); ?></div>
                <div class="history-amt">₹<?= number_format((float)$row['amount'], 2); ?></div>
                <div class="history-date"><?= date('d M Y H:i', strtotime($row['claimed_at'])); ?></div>
            </div>
        <?php endwhile; ?>
    <?php else: ?>
        <div class="history-empty"><i class="fas fa-receipt"></i> No data</div>
    <?php endif; ?>
</div>

<?php if(!empty($message)): ?>
<div class="modal" id="msgModal" style="display:block;">
    <div class="modal-content <?= $messageType ?>">
        <span class="close" id="closeModal">&times;</span>
        <?php if($messageType === 'success'): ?>
            <i class="fas fa-check-circle"></i>
        <?php else: ?>
            <i class="fas fa-times-circle"></i>
        <?php endif; ?>
        <p><?= htmlspecialchars($message) ?></p>
    </div>
</div>
<script>
document.getElementById('closeModal').onclick = function(){
    document.getElementById('msgModal').style.display='none';
};
window.onclick = function(e){
    if(e.target == document.getElementById('msgModal')){
        document.getElementById('msgModal').style.display='none';
    }
};
</script>
<?php endif; ?>

</body>
</html>
