<?php
// API Handler for Game History
// This file handles API communication for game data

header('Content-Type: application/json');
session_start();

if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

require_once "../config.php";

class GameAPIHandler {
    private $apiBaseUrl = "https://api.inout.games/api";
    private $operatorId = "ee2013ed-e1f0-4d6e-97d2-f36619e2eb52";
    private $conn;
    
    public function __construct($dbConnection) {
        $this->conn = $dbConnection;
    }
    
    // Fetch game history from API
    public function fetchGameHistory($userId, $limit = 50, $offset = 0) {
        // In production, this would make actual API calls
        // For now, we'll simulate with realistic data structure
        
        $authToken = $this->getUserAuthToken($userId);
        if (!$authToken) {
            return ['error' => 'No auth token found'];
        }
        
        // Simulate API response structure
        $mockData = [
            'status' => 'success',
            'data' => [
                'games' => [
                    [
                        'id' => 'game_' . uniqid(),
                        'game_name' => 'Chicken Road',
                        'bet_amount' => 100.00,
                        'win_amount' => 250.00,
                        'result' => 'win',
                        'multiplier' => 2.5,
                        'played_at' => date('Y-m-d H:i:s', strtotime('-1 hour')),
                        'game_details' => 'Crossed 5 roads successfully',
                        'status' => 'completed',
                        'game_id' => 'chicken_road',
                        'round_id' => 'round_' . uniqid()
                    ],
                    [
                        'id' => 'game_' . uniqid(),
                        'game_name' => 'Crash Game',
                        'bet_amount' => 50.00,
                        'win_amount' => 0.00,
                        'result' => 'loss',
                        'multiplier' => 0.0,
                        'played_at' => date('Y-m-d H:i:s', strtotime('-2 hours')),
                        'game_details' => 'Crashed at 1.5x',
                        'status' => 'completed',
                        'game_id' => 'crash',
                        'round_id' => 'round_' . uniqid()
                    ],
                    [
                        'id' => 'game_' . uniqid(),
                        'game_name' => 'Dice Roll',
                        'bet_amount' => 25.00,
                        'win_amount' => 75.00,
                        'result' => 'win',
                        'multiplier' => 3.0,
                        'played_at' => date('Y-m-d H:i:s', strtotime('-3 hours')),
                        'game_details' => 'Rolled 6',
                        'status' => 'completed',
                        'game_id' => 'dice',
                        'round_id' => 'round_' . uniqid()
                    ]
                ],
                'total' => 3,
                'page' => 1,
                'per_page' => $limit
            ]
        ];
        
        return $mockData;
    }
    
    // Get user auth token from database
    private function getUserAuthToken($userId) {
        $query = "SELECT api_token FROM users WHERE id = ? LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['api_token'];
        }
        
        return null;
    }
    
    // Save game result to local database
    public function saveGameResult($userId, $gameData) {
        $query = "INSERT INTO api_game_history 
                  (user_id, game_id, game_name, bet_amount, win_amount, result, multiplier, 
                   played_at, api_game_id, round_id, game_details) 
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("issddsdssss", 
            $userId,
            $gameData['game_id'],
            $gameData['game_name'],
            $gameData['bet_amount'],
            $gameData['win_amount'],
            $gameData['result'],
            $gameData['multiplier'],
            $gameData['played_at'],
            $gameData['id'],
            $gameData['round_id'],
            $gameData['game_details']
        );
        
        return $stmt->execute();
    }
    
    // Get local game history
    public function getLocalGameHistory($userId, $limit = 50) {
        $query = "SELECT * FROM api_game_history 
                  WHERE user_id = ? 
                  ORDER BY played_at DESC 
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ii", $userId, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $games = [];
        while ($row = $result->fetch_assoc()) {
            $games[] = $row;
        }
        
        return $games;
    }
    
    // Get game statistics
    public function getGameStats($userId) {
        $query = "SELECT 
                    COUNT(*) as total_games,
                    SUM(CASE WHEN result = 'win' THEN 1 ELSE 0 END) as total_wins,
                    SUM(CASE WHEN result = 'loss' THEN 1 ELSE 0 END) as total_losses,
                    SUM(bet_amount) as total_bet,
                    SUM(win_amount) as total_won,
                    AVG(multiplier) as avg_multiplier,
                    SUM(win_amount - bet_amount) as net_profit
                  FROM api_game_history
                  WHERE user_id = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $handler = new GameAPIHandler($conn);
    
    switch ($action) {
        case 'fetch_history':
            $limit = intval($_POST['limit'] ?? 50);
            $offset = intval($_POST['offset'] ?? 0);
            $data = $handler->fetchGameHistory($_SESSION['id'], $limit, $offset);
            echo json_encode($data);
            break;
            
        case 'get_stats':
            $stats = $handler->getGameStats($_SESSION['id']);
            echo json_encode($stats);
            break;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
            break;
    }
}

// Close connection
$conn->close();
?>
