<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    try {
        // Check if config file exists (use relative path from game folder to root)
        if (!file_exists('../config.php')) {
            exit(json_encode(['status' => 'error', 'message' => 'Config file not found at: ../config.php']));
        }
        
        // Include config file
        require_once '../config.php';
        
        // Check if connection variable exists
        if (!isset($conn)) {
            exit(json_encode(['status' => 'error', 'message' => 'Database connection variable $conn not found in config.php']));
        }
        
        // Check connection
        if ($conn->connect_error) {
            exit(json_encode(['status' => 'error', 'message' => 'Database connection failed: ' . $conn->connect_error]));
        }
        
        // Test database connection
        $testQuery = $conn->query("SELECT 1");
        if (!$testQuery) {
            exit(json_encode(['status' => 'error', 'message' => 'Database query test failed: ' . $conn->error]));
        }
        
        // Get and validate input
        $mobile = trim($_POST['mobile'] ?? '');
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        $refcode = trim($_POST['refcode'] ?? '');

        // Debug: Log received data
        error_log("Registration attempt - Mobile: $mobile, Username: $username");

        // Validation
        if (empty($mobile) || empty($username) || empty($password) || empty($refcode)) {
            exit(json_encode(['status' => 'error', 'message' => 'All fields are required.']));
        }

        if (!preg_match('/^\+91\d{10}$/', $mobile)) {
            exit(json_encode(['status' => 'error', 'message' => 'Invalid mobile number format. Use +91XXXXXXXXXX']));
        }
        
        if (strlen($username) < 3 || strlen($username) > 50) {
            exit(json_encode(['status' => 'error', 'message' => 'Username must be 3-50 characters long.']));
        }
        
        if (strlen($password) < 6) {
            exit(json_encode(['status' => 'error', 'message' => 'Password must be at least 6 characters long.']));
        }
        
        if ($password !== $confirm_password) {
            exit(json_encode(['status' => 'error', 'message' => 'Passwords do not match.']));
        }

        // Check if users table exists
        $tableCheck = $conn->query("SHOW TABLES LIKE 'users'");
        if ($tableCheck->num_rows == 0) {
            exit(json_encode(['status' => 'error', 'message' => 'Users table does not exist']));
        }

        // Check if mobile number already exists
        $mobileCheck = $conn->prepare("SELECT id FROM users WHERE mobile = ?");
        if (!$mobileCheck) {
            exit(json_encode(['status' => 'error', 'message' => 'Mobile check query failed: ' . $conn->error]));
        }
        
        $mobileCheck->bind_param("s", $mobile);
        $mobileCheck->execute();
        $result = $mobileCheck->get_result();

        if ($result->num_rows > 0) {
            exit(json_encode(['status' => 'error', 'message' => 'Mobile number already registered.']));
        }
        $mobileCheck->close();

        // Check if username already exists
        $usernameCheck = $conn->prepare("SELECT id FROM users WHERE username = ?");
        if (!$usernameCheck) {
            exit(json_encode(['status' => 'error', 'message' => 'Username check query failed: ' . $conn->error]));
        }
        
        $usernameCheck->bind_param("s", $username);
        $usernameCheck->execute();
        $result = $usernameCheck->get_result();

        if ($result->num_rows > 0) {
            exit(json_encode(['status' => 'error', 'message' => 'Username already taken.']));
        }
        $usernameCheck->close();

        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

        // Generate unique user code
        $usercode = 'USER' . strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 8));

        // Insert new user
        $insertSql = "INSERT INTO users (username, password, mobile, refcode, usercode, created_at) VALUES (?, ?, ?, ?, ?, NOW())";
        $stmt = $conn->prepare($insertSql);
        
        if (!$stmt) {
            exit(json_encode(['status' => 'error', 'message' => 'Insert query prepare failed: ' . $conn->error]));
        }

        $stmt->bind_param("sssss", $username, $hashedPassword, $mobile, $refcode, $usercode);
        
        if ($stmt->execute()) {
            $userId = $conn->insert_id;
            exit(json_encode([
                'status' => 'success', 
                'message' => 'Account created successfully! Welcome!',
                'user_id' => $userId
            ]));
        } else {
            exit(json_encode(['status' => 'error', 'message' => 'Failed to insert user: ' . $stmt->error]));
        }

    } catch (mysqli_sql_exception $e) {
        exit(json_encode(['status' => 'error', 'message' => 'Database error: ' . $e->getMessage()]));
    } catch (Exception $e) {
        exit(json_encode(['status' => 'error', 'message' => 'Server error: ' . $e->getMessage()]));
    } finally {
        if (isset($conn) && $conn) {
            $conn->close();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Game Portal</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: radial-gradient(circle at top, #1a1a1d, #0d0d0d);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            color: #fff;
            padding: 20px;
        }

        .register-container {
            position: relative;
            z-index: 1;
            background: rgba(20, 20, 22, 0.95);
            padding: 30px;
            border-radius: 20px;
            max-width: 400px;
            width: 100%;
            text-align: center;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.6);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            animation: slideUp 0.6s ease-out;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .register-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            border-radius: 20px;
            padding: 2px;
            background: linear-gradient(45deg, #f7b731, #e09e28, #c88512, #f7b731);
            background-size: 400% 400%;
            animation: gradientShift 4s ease infinite;
            -webkit-mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
            -webkit-mask-composite: xor;
            mask-composite: exclude;
            pointer-events: none;
        }

        @keyframes gradientShift {
            0%, 100% { background-position: 0% 50%; }
            50% { background-position: 100% 50%; }
        }

        .logo {
            width: 80px;
            height: 80px;
            margin: 0 auto 20px;
            background: linear-gradient(135deg, #f7b731, #e09e28);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2rem;
        }

        h1 {
            color: #f7b731;
            margin-bottom: 25px;
            font-size: 1.8rem;
            font-weight: 700;
        }

        .form-group {
            position: relative;
            margin-bottom: 20px;
            text-align: left;
        }

        .form-group input {
            width: 100%;
            padding: 15px 50px 15px 15px;
            border: 2px solid rgba(255, 255, 255, 0.1);
            border-radius: 12px;
            background: rgba(255, 255, 255, 0.05);
            color: #fff;
            font-size: 16px;
            transition: all 0.3s ease;
            outline: none;
        }

        .form-group input:focus {
            border-color: #f7b731;
            background: rgba(247, 183, 49, 0.1);
            box-shadow: 0 0 0 3px rgba(247, 183, 49, 0.2);
        }

        .form-group input::placeholder {
            color: rgba(255, 255, 255, 0.6);
        }

        .form-group .icon {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #f7b731;
            font-size: 1.2rem;
            cursor: pointer;
        }

        .register-btn {
            width: 100%;
            padding: 15px;
            border: none;
            border-radius: 12px;
            background: linear-gradient(135deg, #f7b731, #e09e28);
            color: #000;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 10px;
        }

        .register-btn:hover {
            transform: translateY(-2px);
            background: linear-gradient(135deg, #e09e28, #c88512);
        }

        .login-link {
            margin-top: 20px;
            color: rgba(255, 255, 255, 0.7);
            font-size: 14px;
        }

        .login-link a {
            color: #f7b731;
            text-decoration: none;
            font-weight: 500;
        }

        .login-link a:hover {
            color: #e09e28;
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="register-container">
        <div class="logo">🎮</div>
        <h1>Create Account</h1>
        
        <form id="registerForm" method="POST">
            <div class="form-group">
                <input type="text" id="mobile" name="mobile" placeholder="Mobile Number (+91)" required maxlength="13">
                <i class="fas fa-phone icon"></i>
            </div>

            <div class="form-group">
                <input type="text" id="username" name="username" placeholder="Username" required maxlength="50">
                <i class="fas fa-user icon"></i>
            </div>

            <div class="form-group">
                <input type="password" id="password" name="password" placeholder="Password (min 6 chars)" required minlength="6">
                <i class="fas fa-eye icon" id="togglePassword1" onclick="togglePassword('password', 'togglePassword1')"></i>
            </div>

            <div class="form-group">
                <input type="password" id="confirm_password" name="confirm_password" placeholder="Confirm Password" required minlength="6">
                <i class="fas fa-eye icon" id="togglePassword2" onclick="togglePassword('confirm_password', 'togglePassword2')"></i>
            </div>

            <div class="form-group">
                <input type="text" name="refcode" placeholder="Referral Code" required>
                <i class="fas fa-gift icon"></i>
            </div>

            <button type="submit" class="register-btn">
                <i class="fas fa-user-plus"></i> Create Account
            </button>
        </form>

        <div class="login-link">
            Already have an account? <a href="login.php">Login here</a>
        </div>
    </div>

    <script>
        // Mobile number formatting
        document.getElementById('mobile').addEventListener('input', function() {
            let value = this.value.replace(/[^\d+]/g, '');
            
            if (!value.startsWith('+91')) {
                if (value.startsWith('91')) {
                    value = '+' + value;
                } else if (value.startsWith('+')) {
                    value = '+91' + value.replace(/^\+/, '');
                } else {
                    value = '+91' + value;
                }
            }
            
            if (value.length > 13) {
                value = value.substring(0, 13);
            }
            
            this.value = value;
        });

        // Toggle password visibility
        function togglePassword(fieldId, iconId) {
            const field = document.getElementById(fieldId);
            const icon = document.getElementById(iconId);
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        // Form submission with better error handling
        document.getElementById('registerForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            // Show loading
            Swal.fire({
                title: 'Creating Account...',
                allowOutsideClick: false,
                showConfirmButton: false,
                background: '#1a1a1d',
                color: '#fff',
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            // Submit form
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => {
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers);
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                return response.text();
            })
            .then(text => {
                console.log('Response text:', text);
                
                try {
                    return JSON.parse(text);
                } catch (e) {
                    console.error('JSON parse error:', e);
                    console.error('Response was:', text);
                    throw new Error('Invalid JSON response from server. Check console for details.');
                }
            })
            .then(data => {
                console.log('Parsed data:', data);
                
                if (data.status === 'success') {
                    Swal.fire({
                        icon: 'success',
                        title: 'Success!',
                        text: data.message,
                        background: '#1a1a1d',
                        color: '#fff',
                        confirmButtonColor: '#f7b731'
                    }).then(() => {
                        window.location.href = 'login.php';
                    });
                } else {
                    Swal.fire({
                        icon: 'error',
                        title: 'Registration Failed',
                        text: data.message,
                        background: '#1a1a1d',
                        color: '#fff',
                        confirmButtonColor: '#f7b731'
                    });
                }
            })
            .catch(error => {
                console.error('Fetch error:', error);
                Swal.fire({
                    icon: 'error',
                    title: 'Connection Error',
                    text: error.message + ' - Check browser console for details',
                    background: '#1a1a1d',
                    color: '#fff',
                    confirmButtonColor: '#f7b731'
                });
            });
        });
    </script>
</body>
</html>