<?php
session_start();
$isLoggedIn = isset($_SESSION["loggedin"]) && $_SESSION["loggedin"] === true;

if (!$isLoggedIn) {
    header("Location: /game/login");
    exit();
}

require_once "config.php";

// Validate session username
if (!isset($_SESSION['username']) || empty($_SESSION['username'])) {
    session_destroy();
    header("Location: /game/login");
    exit();
}

$username = $_SESSION['username'];

// Fetch user's balance
$stmt = $conn->prepare("SELECT balance FROM users WHERE username = ?");
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$balance = $user['balance'] ?? 0;
$stmt->close();

// Fetch transactions (assuming a 'transactions' table with columns: id, user_id, type, amount, description, created_at)
// Types: deposit, withdrawal, win, loss, bonus, referral, etc.
$transactions = [];
$query = "SELECT t.id, t.type, t.amount, t.description, t.created_at 
          FROM transactions t 
          INNER JOIN users u ON t.user_id = u.id 
          WHERE u.username = ? 
          ORDER BY t.created_at DESC";
$stmt = $conn->prepare($query);
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $transactions[] = $row;
}
$stmt->close();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Chicken Road - Transactions</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Poppins', sans-serif;
            background: #0c0c0f;
            color: #fff;
            overflow-x: hidden;
            padding-bottom: 100px;
        }

        /* ===== Header ===== */
        header {
            position: sticky;
            top: 0;
            z-index: 200;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: rgba(25, 25, 30, 0.75);
            backdrop-filter: blur(12px);
            border-bottom: 1px solid rgba(255,255,255,0.1);
            animation: slideDown .5s ease;
        }

        @keyframes slideDown {
            from { transform: translateY(-100%); }
            to { transform: translateY(0); }
        }

        .logo {
            font-size: 1.4rem;
            font-weight: 700;
            background: linear-gradient(90deg, #ffd65a, #fbb034);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            letter-spacing: 1px;
        }

        .balance {
            background: rgba(255,215,0,0.08);
            border: 1px solid rgba(255,215,0,0.3);
            padding: 6px 14px;
            border-radius: 20px;
            font-weight: 600;
            font-size: .95rem;
            color: #ffd65a;
        }

        .balance i {
            color: #ffd65a;
        }

        /* ===== Filters Section ===== */
        .filters {
            max-width: 800px;
            margin: 20px auto;
            padding: 0 20px;
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            justify-content: center;
        }

        .filter-input {
            padding: 10px 15px;
            border-radius: 20px;
            border: 1px solid rgba(255,215,0,0.3);
            background: rgba(255,255,255,0.05);
            color: #fff;
            outline: none;
            transition: all 0.3s ease;
            width: 180px;
        }

        .filter-input:focus {
            border-color: #ffd65a;
            box-shadow: 0 0 10px rgba(255,215,0,0.2);
        }

        select.filter-input {
            appearance: none;
            background-image: url('data:image/svg+xml;utf8,<svg fill="%23ffd65a" height="24" viewBox="0 0 24 24" width="24" xmlns="http://www.w3.org/2000/svg"><path d="M7 10l5 5 5-5z"/><path d="M0 0h24v24H0z" fill="none"/></svg>');
            background-repeat: no-repeat;
            background-position: right 10px top 50%;
            background-size: 20px;
        }

        .filter-btn {
            background: linear-gradient(135deg, #ffd65a, #fbb034);
            color: #111;
            padding: 10px 20px;
            border-radius: 20px;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .filter-btn:hover {
            background: linear-gradient(135deg, #fbb034, #e09e28);
        }

        /* ===== Transactions Table ===== */
        .transactions-container {
            max-width: 800px;
            margin: 0 auto 30px;
            padding: 0 20px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background: rgba(255,255,255,0.06);
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 6px 18px rgba(0,0,0,0.3);
        }

        th, td {
            padding: 12px 15px;
            text-align: left;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }

        th {
            background: rgba(255,215,0,0.1);
            color: #ffd65a;
            font-weight: 600;
        }

        tr:hover {
            background: rgba(255,215,0,0.08);
        }

        .amount-positive {
            color: #4caf50;
        }

        .amount-negative {
            color: #f44336;
        }

        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 20px;
        }

        .page-btn {
            background: rgba(255,215,0,0.1);
            color: #ffd65a;
            padding: 8px 12px;
            border-radius: 20px;
            text-decoration: none;
            transition: all 0.3s ease;
        }

        .page-btn.active, .page-btn:hover {
            background: #ffd65a;
            color: #111;
        }

        /* ===== Bottom Nav ===== */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 60px;
            backdrop-filter: blur(12px);
            background: rgba(20,20,20,0.7);
            display: flex;
            justify-content: space-around;
            align-items: center;
            border-top: 1px solid rgba(255,255,255,0.1);
            z-index: 100;
        }

        .bottom-nav a {
            flex: 1;
            text-align: center;
            color: #aaa;
            font-size: 12px;
            text-decoration: none;
            display: flex;
            flex-direction: column;
            align-items: center;
            transition: color .3s ease;
        }

        .bottom-nav a i {
            font-size: 18px;
            margin-bottom: 3px;
        }

        .bottom-nav a.active, .bottom-nav a:hover {
            color: #ffd65a;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .filters {
                flex-direction: column;
                align-items: center;
            }

            table {
                font-size: 0.9rem;
            }

            th, td {
                padding: 10px;
            }
        }
    </style>
</head>
<body>

<!-- Header -->
<header>
    <div class="logo">Chicken Road</div>
    <div class="balance">
        <i class="fas fa-wallet"></i> ₹<span id="balance"><?php echo number_format($balance, 2); ?></span>
    </div>
</header>

<!-- Filters -->
<div class="filters">
    <input type="date" id="start-date" class="filter-input" placeholder="Start Date">
    <input type="date" id="end-date" class="filter-input" placeholder="End Date">
    <select id="type-filter" class="filter-input">
        <option value="">All Types</option>
        <option value="deposit">Deposit</option>
        <option value="withdrawal">Withdrawal</option>
        <option value="win">Win</option>
        <option value="loss">Loss</option>
        <option value="bonus">Bonus</option>
        <option value="referral">Referral</option>
    </select>
    <input type="text" id="search-desc" class="filter-input" placeholder="Search Description">
    <button class="filter-btn" onclick="applyFilters()">Apply Filters</button>
</div>

<!-- Transactions Table -->
<div class="transactions-container">
    <table id="transactions-table">
        <thead>
            <tr>
                <th>Date</th>
                <th>Type</th>
                <th>Amount</th>
                <th>Description</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($transactions as $tx): ?>
                <tr>
                    <td><?php echo date('Y-m-d H:i', strtotime($tx['created_at'])); ?></td>
                    <td><?php echo ucfirst($tx['type']); ?></td>
                    <td class="<?php echo ($tx['amount'] >= 0) ? 'amount-positive' : 'amount-negative'; ?>">
                        ₹<?php echo number_format($tx['amount'], 2); ?>
                    </td>
                    <td><?php echo htmlspecialchars($tx['description']); ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <!-- Pagination (Client-side for simplicity; can be server-side for large datasets) -->
    <div class="pagination" id="pagination"></div>
</div>

<!-- Bottom Navigation -->
<nav class="bottom-nav">
    <a href="/home"><i class="fas fa-gamepad"></i>Games</a>
    <a href="/game/invite"><i class="fas fa-user-friends"></i>Invite</a>
    <a href="/game/wallet"><i class="fas fa-wallet"></i>Wallet</a>
    <a href="/game/profile"><i class="fas fa-user"></i>Profile</a>
    <a href="/transactions.php" class="active"><i class="fas fa-history"></i>Transactions</a>
</nav>

<!-- Scripts -->
<script>
const transactions = <?php echo json_encode($transactions); ?>;
const rowsPerPage = 10;
let currentPage = 1;
let filteredTransactions = [...transactions];

function renderTable(data) {
    const tbody = document.querySelector('#transactions-table tbody');
    tbody.innerHTML = '';
    data.forEach(tx => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${new Date(tx.created_at).toLocaleString()}</td>
            <td>${tx.type.charAt(0).toUpperCase() + tx.type.slice(1)}</td>
            <td class="${tx.amount >= 0 ? 'amount-positive' : 'amount-negative'}">
                ₹${parseFloat(tx.amount).toFixed(2)}
            </td>
            <td>${tx.description}</td>
        `;
        tbody.appendChild(row);
    });
}

function renderPagination(totalPages) {
    const pagination = document.getElementById('pagination');
    pagination.innerHTML = '';
    for (let i = 1; i <= totalPages; i++) {
        const btn = document.createElement('a');
        btn.href = '#';
        btn.classList.add('page-btn');
        if (i === currentPage) btn.classList.add('active');
        btn.textContent = i;
        btn.addEventListener('click', (e) => {
            e.preventDefault();
            currentPage = i;
            paginate();
        });
        pagination.appendChild(btn);
    }
}

function paginate() {
    const start = (currentPage - 1) * rowsPerPage;
    const end = start + rowsPerPage;
    const pageData = filteredTransactions.slice(start, end);
    renderTable(pageData);
    const totalPages = Math.ceil(filteredTransactions.length / rowsPerPage);
    renderPagination(totalPages);
}

function applyFilters() {
    const startDate = document.getElementById('start-date').value;
    const endDate = document.getElementById('end-date').value;
    const type = document.getElementById('type-filter').value.toLowerCase();
    const search = document.getElementById('search-desc').value.toLowerCase();

    filteredTransactions = transactions.filter(tx => {
        const txDate = new Date(tx.created_at);
        const matchesDate = (!startDate || txDate >= new Date(startDate)) && (!endDate || txDate <= new Date(endDate));
        const matchesType = !type || tx.type.toLowerCase() === type;
        const matchesSearch = !search || tx.description.toLowerCase().includes(search);
        return matchesDate && matchesType && matchesSearch;
    });

    currentPage = 1;
    paginate();
}

// Initial render
paginate();

// Balance Auto Refresh
setInterval(() => {
    fetch('/game/balance')
        .then(res => res.json())
        .then(data => {
            if (data.balance !== undefined) {
                document.getElementById('balance').textContent = data.balance;
            }
        })
        .catch(err => console.error('Balance fetch error:', err));
}, 10000);
</script>
</body>
</html>
