<?php
session_start([
  'cookie_httponly' => true,
  'cookie_secure' => true,
  'use_strict_mode' => true
]);

if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: login");
    exit;
}

require_once "../config.php";

// Helper function for SQL sum queries
function fetchSum($conn, $query, $params) {
    $stmt = $conn->prepare($query);
    if ($stmt === false) return 0;
    $types = str_repeat('s', count($params));
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    $sum = $result->fetch_assoc();
    $stmt->close();
    return ($sum['total'] == "") ? 0 : round($sum['total'], 2);
}

$balance = fetchSum($conn, "SELECT balance as total FROM users WHERE username=?", [$_SESSION['username']]);
$totalDeposits = fetchSum($conn, "SELECT SUM(amount) as total FROM deposits WHERE usercode=? AND status='completed'", [$_SESSION['usercode']]);
$totalWithdrawals = fetchSum($conn, "SELECT SUM(amount) as total FROM withdrawals WHERE usercode=? AND status='completed'", [$_SESSION['usercode']]);

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Wallet - Chicken Road</title>
  <link href="https://fonts.googleapis.com/css2?family=Montserrat:wght@400;500;600;700&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css" />
  <style>
    :root {
      --primary: #fbb034;
      --secondary: #fbb034;
      --dark-bg: #0c0c0f;
      --card-bg: #131317;
      --highlight: rgba(255, 215, 0, 0.1);
      --text-light: #e0e0e0;
      --text-muted: #aaa;
    }

    * {
      box-sizing: border-box;
      margin: 0;
      padding: 0;
    }

    body {
      font-family: 'Montserrat', sans-serif;
      background: var(--dark-bg);
      color: var(--text-light);
      margin: 0;
      padding: 80px 0 80px;
      min-height: 100vh;
    }

    header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 15px 20px;
      background: linear-gradient(135deg, rgba(30, 30, 35, 0.95) 0%, rgba(20, 20, 25, 0.95) 100%);
      backdrop-filter: blur(10px);
      -webkit-backdrop-filter: blur(10px);
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      z-index: 1000;
      border-bottom: 1px solid rgba(255, 255, 255, 0.1);
      height: 60px;
      box-shadow: 0 5px 20px rgba(0, 0, 0, 0.3);
    }

    .logo {
      font-size: 1.4rem;
      font-weight: 700;
      color: var(--primary);
      text-shadow: 0 0 10px rgba(251, 176, 52, 0.3);
    }

    .balance {
      background: linear-gradient(135deg, rgba(251, 176, 52, 0.2) 0%, rgba(247, 151, 30, 0.2) 100%);
      border-radius: 20px;
      color: var(--primary);
      padding: 8px 16px;
      display: flex;
      align-items: center;
      gap: 8px;
      font-weight: 600;
      box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
      border: 1px solid rgba(251, 176, 52, 0.2);
    }

    .wallet-container {
      max-width: 600px;
      margin: 0 auto;
      padding: 20px;
      animation: fadeIn 0.5s ease-out;
    }

    .wallet-card {
      background: linear-gradient(135deg, rgba(19, 19, 23, 0.9) 0%, rgba(25, 25, 30, 0.9) 100%);
      border-radius: 16px;
      padding: 25px;
      margin-bottom: 25px;
      box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
      border: 1px solid rgba(255, 255, 255, 0.05);
      backdrop-filter: blur(5px);
      -webkit-backdrop-filter: blur(5px);
      position: relative;
      overflow: hidden;
    }

    .wallet-card::before {
      content: '';
      position: absolute;
      top: -50%;
      left: -50%;
      width: 200%;
      height: 200%;
      background: radial-gradient(circle, rgba(255, 255, 255, 0.1) 0%, transparent 70%);
      transform: rotate(30deg);
    }

    .wallet-header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 20px;
    }

    .wallet-title {
      font-size: 1.2rem;
      font-weight: 600;
      color: var(--primary);
    }

    .refresh-btn {
      background: rgba(251, 176, 52, 0.1);
      border: 1px solid rgba(251, 176, 52, 0.3);
      color: var(--primary);
      padding: 8px 12px;
      border-radius: 8px;
      font-size: 0.9rem;
      display: flex;
      align-items: center;
      gap: 6px;
      cursor: pointer;
      transition: all 0.3s;
    }

    .refresh-btn:hover {
      background: rgba(251, 176, 52, 0.2);
    }

    .wallet-amount {
      font-size: 2.2rem;
      font-weight: 700;
      color: var(--primary);
      margin-bottom: 5px;
      text-align: center;
    }

    .wallet-subtitle {
      color: var(--text-muted);
      text-align: center;
      margin-bottom: 25px;
      font-size: 0.9rem;
    }

    .stats-row {
      display: flex;
      gap: 15px;
      margin-bottom: 20px;
      justify-content: space-between;
    }

    .stat-card {
      flex: 1;
      background: rgba(0, 0, 0, 0.2);
      border-radius: 12px;
      padding: 15px;
      text-align: center;
      border: 1px solid rgba(255, 255, 255, 0.05);
      min-width: 48%;
    }

    .stat-title {
      font-size: 0.8rem;
      color: var(--text-muted);
      margin-bottom: 8px;
    }

    .stat-value {
      font-size: 1.1rem;
      font-weight: 600;
      color: var(--secondary);
    }

    .action-list {
      display: flex;
      gap: 10px;
      margin: 25px 0;
      overflow-x: auto;
      padding-bottom: 10px;
      scrollbar-width: none;
      justify-content: space-between;
    }

    .action-list::-webkit-scrollbar {
      display: none;
    }

    .action-card {
      background: transparent;
      padding: 12px 10px;
      border-radius: 12px;
      text-align: center;
      cursor: pointer;
      width: 23%;
      border: 1px solid rgba(255, 255, 255, 0.05);
    }



    .action-icon {
      width: 24px;
      height: 24px;
      margin: 0 auto 6px;
      display: block;
    }

    .action-title {
      font-size: 0.7rem;
      font-weight: 500;
      color: var(--text-light);
      white-space: nowrap;
    }

    .bottom-nav {
      position: fixed;
      bottom: 0;
      left: 0;
      right: 0;
      background: linear-gradient(135deg, rgba(30, 30, 35, 0.95) 0%, rgba(20, 20, 25, 0.95) 100%);
      backdrop-filter: blur(10px);
      -webkit-backdrop-filter: blur(10px);
      border-top: 1px solid rgba(255, 255, 255, 0.1);
      display: flex;
      justify-content: space-around;
      height: 70px;
      z-index: 1000;
      box-shadow: 0 -5px 20px rgba(0, 0, 0, 0.3);
    }

    .bottom-nav a {
      flex: 1;
      text-align: center;
      color: var(--text-muted);
      font-size: 0.7rem;
      text-decoration: none;
      padding: 12px 0 8px;
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: center;
      transition: all 0.2s ease;
    }

    .bottom-nav a i {
      font-size: 1.2rem;
      margin-bottom: 5px;
      transition: all 0.2s ease;
    }

    .bottom-nav a.active {
      color: var(--primary);
    }

    .bottom-nav a.active i {
      transform: translateY(-5px);
      text-shadow: 0 5px 10px rgba(251, 176, 52, 0.3);
    }

    @media screen and (max-width: 600px) {
      .wallet-container {
        padding: 20px 15px;
      }

      .wallet-card {
        padding: 20px;
      }

      .action-card {
        min-width: 23%;
        padding: 10px 8px;
      }

      .action-title {
        font-size: 0.6rem;
      }
    }

    /* Animations */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(20px);
      }

      to {
        opacity: 1;
        transform: translateY(0);
      }
    }
  </style>
  <?php include "wallet-style.css"; // if you use separate file, otherwise keep the full style here ?>
  </style>
</head>

<body>

  <header>
    <div class="logo">Chicken Road</div>
    <div class="balance">
      <i class="fas fa-wallet"></i>
      ₹<span id="balance">
        <?php echo htmlspecialchars(number_format($balance, 2)); ?>
      </span>
    </div>
  </header>

  <div class="wallet-container">
    <div class="wallet-card">
      <div class="wallet-header">
        <div class="wallet-title">Wallet Balance</div>
        <button class="refresh-btn" id="refreshBtn">
          <i class="fas fa-sync-alt"></i>
          Refresh
        </button>
      </div>

      <div class="wallet-amount">₹
        <?php echo htmlspecialchars(number_format($balance, 2)); ?>
      </div>
      <div class="wallet-subtitle">Total available balance</div>

      <div class="stats-row">
        <div class="stat-card">
          <div class="stat-title">Total Deposits</div>
          <div class="stat-value">₹
            <?php echo htmlspecialchars(number_format($totalDeposits, 2)); ?>
          </div>
        </div>
        <div class="stat-card">
          <div class="stat-title">Total Withdrawals</div>
          <div class="stat-value">₹
            <?php echo htmlspecialchars(number_format($totalWithdrawals, 2)); ?>
          </div>
        </div>
      </div>

      <!-- ✅ Actions moved inside wallet-card -->
      <div class="action-list" style="margin-top: 20px;">
        <div class="action-card" onclick="window.location.href='/recharge'">
          <img src="/images/rechargeIcon.png" class="action-icon" alt="Deposit">
          <div class="action-title">Deposit</div>
        </div>
         
        <div class="action-card" onclick="window.location.href='/withdraw'">
          <img src="https://bdg55.com/assets/png/widthdrawBlue-5fcf62bd.webp" class="action-icon" alt="Withdraw">
          <div class="action-title">Withdraw</div>
        </div>
        <div class="action-card" onclick="window.location.href='/deposithistory'">
          <img src="/images/rechargeHistory.png" class="action-icon"
            alt="Deposit History">
          <div class="action-title">
            <span>Deposit</span><br>
            <span>History</span>
          </div>
        </div>
        <div class="action-card" onclick="window.location.href='/withdrawalhistory'">
          <img src="/images/withdrawHistory1.png" class="action-icon"
            alt="Withdrawal History">
          <div class="action-title">
            <span>Withdrawal</span><br>
            <span>History</span>
          </div>
        </div>
      </div>
      <!-- End Actions -->
    </div>
  </div>

  <!-- ✅ Bottom navigation remains unchanged -->
  <nav class="bottom-nav">
    <a href="/home"><i class="fas fa-gamepad"></i>Games</a>
    <a href="/game/invite"><i class="fas fa-user-friends"></i>Invite</a>
    <a href="/game/wallet" class="active"><i class="fas fa-wallet"></i>Wallet</a>
    <a href="/game/profile"><i class="fas fa-user"></i>Account</a>
  </nav>

  <script>
    // Balance update animation
    function animateBalanceUpdate(oldVal, newVal) {
      const balanceElement = document.getElementById('balance');
      const walletAmount = document.querySelector('.wallet-amount');
      const duration = 1000;
      const start = parseFloat(oldVal.replace(/,/g, ''));
      const end = parseFloat(newVal.replace(/,/g, ''));
      const startTime = performance.now();

      function updateBalance(timestamp) {
        const progress = Math.min((timestamp - startTime) / duration, 1);
        const currentValue = start + (end - start) * progress;
        const formattedValue = currentValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
        balanceElement.textContent = formattedValue;
        walletAmount.textContent = '₹' + formattedValue;

        if (progress < 1) {
          requestAnimationFrame(updateBalance);
        }
      }

      requestAnimationFrame(updateBalance);
    }

    // Refresh button functionality
    document.getElementById('refreshBtn').addEventListener('click', function () {
      fetch('/balance.php', { credentials: 'same-origin' })
        .then(r => r.json())
        .then(d => {
          if (d.balance !== undefined) {
            const currentBalance = document.getElementById('balance').textContent;
            const newBalance = parseFloat(d.balance).toFixed(2);
            if (currentBalance !== newBalance) {
              animateBalanceUpdate(currentBalance, newBalance);
            }
          }
        })
        .catch(err => console.error('Error fetching balance:', err));
    });

    // Auto update balance every 10s
    setInterval(() => {
      fetch('/balance.php', { credentials: 'same-origin' })
        .then(r => r.json())
        .then(d => {
          if (d.balance !== undefined) {
            const currentBalance = document.getElementById('balance').textContent;
            const newBalance = parseFloat(d.balance).toFixed(2);
            if (currentBalance !== newBalance) {
              animateBalanceUpdate(currentBalance, newBalance);
            }
          }
        })
        .catch(err => console.error('Error fetching balance:', err));
    }, 10000);
  </script>

</body>

</html>